#include "MTFile.h"
#include "IFileStream.h"
#include "IErrors.h"
#include "MTExceptions.h"
#include "MTUtilities.h"

#include <fstream>

//	MTFile::MTFile()
//	
//	constructor
MTFile::MTFile()
{
	
}

//	MTFile::~MTFile()
//	
//	destructor
MTFile::~MTFile()
{
	
}

//	void MTFile::LoadFile(FSSpec * theFile)
//	
//	loads a file
//	
//	theFile = pointer to FSSpec
void MTFile::LoadFile(FSSpec * theFile)
{
	IFileStream	stream;
	
	if(pstrcmp(theFile->name, "\ppersist.dat") == 0)	// baka na user check
		throw MTException(	"OniTools only edits Oni level data files, not save game files. "
							"Try OSGE (http://oni.bungie.org/onishots/OSGE/osge.htm) for your save editing needs.");
	
	baseFile = *theFile;
	
	BuildOtherFiles();
	
	stream.OpenFile(&baseFile);
	
	ParseHeader(&stream);
	ParseFileList(&stream);
	
	stream.CloseFile();
}

void MTFile::ExportFile(UInt32 offset, UInt32 size)
{
	StandardFileReply	reply;
	OSErr				theErr;
	
	StandardPutFile("\pExport File:", "\pData Segment", &reply);
	
	if(reply.sfGood)
	{
		if(reply.sfReplacing)
		{
			theErr = FSpDelete(&reply.sfFile);
			if(theErr)
				throw MTOSFileException(theErr, "Error deleting old file");
		}
		
		theErr = FSpCreate(&reply.sfFile, 'ONIt', 'DATA', reply.sfScript);
		if(theErr)
			throw MTOSFileException(theErr, "Error creating new file");
		
		IFileStream	inStream, outStream;
		
		inStream.OpenFile(&baseFile);
		outStream.OpenFile(&reply.sfFile);
		
		inStream.SetPosition(offset);
		
		CopyToFile(&inStream, &outStream, size);
		
		inStream.CloseFile();
		outStream.CloseFile();
	}
}

//	void MTFile::ExportFileByIdx(UInt32 idx)
//	
//	exports a file referenced by index
//	
//	idx = index
void MTFile::ExportFileByIdx(UInt32 idx)
{
	StandardFileReply	reply;
	char				buf[256];
	Str255				theString;
	OSErr				theErr;
	
	if(idx >= fileList.size())
		return;
	
	if(fileList[idx].name.empty())
	{
		std::sprintf(buf, "unnamed.%.4s", &fileList[idx].type);
	}
	else
	{
		std::sprintf(buf, "%s.%.4s", fileList[idx].name.c_str(), &fileList[idx].type);
	}
	
	CopyCStringToPascal(buf, theString);
	
	StandardPutFile("\pExport File:", theString, &reply);
	
	if(reply.sfGood)
	{
		if(reply.sfReplacing)
		{
			theErr = FSpDelete(&reply.sfFile);
			if(theErr)
				throw MTOSFileException(theErr, "Error deleting old file");
		}
		
		theErr = FSpCreate(&reply.sfFile, 'ONIt', fileList[idx].type, reply.sfScript);
		if(theErr)
			throw MTOSFileException(theErr, "Error creating new file");
		
		IFileStream	inStream, outStream;
		
		inStream.OpenFile(&baseFile);
		outStream.OpenFile(&reply.sfFile);
		
		inStream.SetPosition(fileList[idx].start);
		
		CopyToFile(&inStream, &outStream, fileList[idx].size);
		
		inStream.CloseFile();
		outStream.CloseFile();
	}
}

//	void MTFile::ImportFileByIdx(UInt32 idx)
//	
//	imports a file referenced by index
//	
//	idx = file index
void MTFile::ImportFileByIdx(UInt32 idx)
{
	IFileStream	inStream;
	IFileStream	* outStream = nil;
	
	try
	{
		FSSpec		theFile;
		
		if(NavigationGetFileCustom(&theFile, "\pPlease choose the file you want to import."))
		{
			inStream.OpenFile(&theFile);
			outStream = OpenBaseFile();
			
			if(!outStream)
				throw MTException("Error opening file.");
			
			if(inStream.GetLength() != fileList[idx].size)
				throw MTException("File sizes don't match.");
			
			outStream->SetPosition(fileList[idx].start);
			
			CopyToFile(&inStream, outStream, fileList[idx].size);
			
			inStream.CloseFile();
			outStream->CloseFile();
			
			delete outStream;
		}
	}
	catch(...)
	{
		if(outStream)
			delete outStream;
		
		throw;
	}
}

//	UInt8 * MTFile::LoadFileByIdx(UInt32 idx)
//	
//	loads a file referenced by index
//	
//	idx = file index
//	returns buffer
UInt8 * MTFile::LoadFileByIdx(UInt32 idx)
{
	UInt8		* buf;
	IFileStream	inStream;
	
	if(idx >= fileList.size())
		return nil;
	
	buf = (UInt8 *)NewPtr(fileList[idx].size);
	if(!buf)
		throw MTMemoryException("Out of memory");
	
	inStream.OpenFile(&baseFile);
	
	inStream.SetPosition(fileList[idx].start);
	inStream.ReadBuffer(buf, fileList[idx].size);
	
	inStream.CloseFile();
	
	return buf;
}

//	void MTFile::ExportFileByID(UInt32 id)
//	
//	exports a file referenced by file id
//	
//	id = embedded file id
void MTFile::ExportFileByID(UInt32 id)
{
	UInt32	idx = LookupIdxFromID(id);
	
	if(idx != 0xFFFFFFFF)
		ExportFileByIdx(idx);
}

//	UInt8 * MTFile::LoadFileByID(UInt32 id)
//	
//	loads a file referenced by file id
//	
//	id = embedded file id
//	returns buffer
UInt8 * MTFile::LoadFileByID(UInt32 id)
{
	UInt32	idx = LookupIdxFromID(id);
	
	if(idx != 0xFFFFFFFF)
		return LoadFileByIdx(idx);
	
	return nil;
}

//	UInt8 * MTFile::LoadBase(UInt32 offset, UInt32 size)
//	
//	loads a section of the base file
//	
//	offset = file offset
//	size = size
//	returns buffer
UInt8 * MTFile::LoadBase(UInt32 offset, UInt32 size)
{
	UInt8		* buf;
	IFileStream	inStream;
	
	buf = (UInt8 *)NewPtr(size);
	if(!buf)
		throw MTMemoryException("Out of memory");
	
	inStream.OpenFile(&baseFile);
	
	inStream.SetPosition(offset);
	inStream.ReadBuffer(buf, size);
	
	inStream.CloseFile();
	
	return buf;
}

//	IFileStream * MTFile::OpenBaseFile(void)
//	
//	opens the database file for streaming
//	
//	returns the stream
IFileStream * MTFile::OpenBaseFile(void)
{
	IFileStream	* stream = new IFileStream;
	
	stream->OpenFile(&baseFile);
	
	return stream;
}

//	void MTFile::ExportSegment(UInt32 offset, UInt32 size)
//	
//	exports a section of the data file
//	
//	offset = file offset
//	size = size
void MTFile::ExportSegment(UInt32 offset, UInt32 size)
{
	StandardFileReply	reply;
	OSErr				theErr;
	
	StandardPutFile("\pExport File:", "\pUntitled Document", &reply);
	
	if(reply.sfGood)
	{
		if(reply.sfReplacing)
		{
			theErr = FSpDelete(&reply.sfFile);
			if(theErr)
				throw MTOSFileException(theErr, "Error deleting old file");
		}
		
		theErr = FSpCreate(&reply.sfFile, 'ONIt', 'DATA', reply.sfScript);
		if(theErr)
			throw MTOSFileException(theErr, "Error creating new file");
		
		IFileStream	inStream, outStream;
		
		inStream.OpenFile(&dataFile);
		outStream.OpenFile(&reply.sfFile);
		
		inStream.SetPosition(offset);
		
		CopyToFile(&inStream, &outStream, size);
		
		inStream.CloseFile();
		outStream.CloseFile();
	}
}

//	UInt8 * MTFile::LoadSegment(UInt32 offset, UInt32 size)
//	
//	loads a section of the data file
//	
//	offset = file offset
//	size = size
//	returns buffer
UInt8 * MTFile::LoadSegment(UInt32 offset, UInt32 size)
{
	UInt8		* buf;
	IFileStream	inStream;
	
	buf = (UInt8 *)NewPtr(size);
	if(!buf)
		throw MTMemoryException("Out of memory");
	
	inStream.OpenFile(&dataFile);
	
	inStream.SetPosition(offset);
	inStream.ReadBuffer(buf, size);
	
	inStream.CloseFile();
	
	return buf;
}

//	IFileStream * MTFile::OpenDataFile(void)
//	
//	opens the data file for streaming
//	
//	returns the stream
IFileStream * MTFile::OpenDataFile(void)
{
	IFileStream	* stream = new IFileStream;
	
	stream->OpenFile(&dataFile);
	
	return stream;
}

//	void MTFile::ExportRaw(UInt32 offset, UInt32 size)
//	
//	exports a section of the raw file
//	
//	offset = file offset
//	size = size
void MTFile::ExportRaw(UInt32 offset, UInt32 size)
{
	StandardFileReply	reply;
	OSErr				theErr;
	
	StandardPutFile("\pExport File:", "\pUntitled Document", &reply);
	
	if(reply.sfGood)
	{
		if(reply.sfReplacing)
		{
			theErr = FSpDelete(&reply.sfFile);
			if(theErr)
				throw MTOSFileException(theErr, "Error deleting old file");
		}
		
		theErr = FSpCreate(&reply.sfFile, 'ONIt', 'DATA', reply.sfScript);
		if(theErr)
			throw MTOSFileException(theErr, "Error creating new file");
		
		IFileStream	inStream, outStream;
		
		inStream.OpenFile(&rawFile);
		outStream.OpenFile(&reply.sfFile);
		
		inStream.SetPosition(offset);
		
		CopyToFile(&inStream, &outStream, size);
		
		inStream.CloseFile();
		outStream.CloseFile();
	}
}

//	UInt8 * MTFile::LoadRaw(UInt32 offset, UInt32 size)
//	
//	loads a section of the raw file
//	
//	offset = file offset
//	size = size
//	returns buffer
UInt8 * MTFile::LoadRaw(UInt32 offset, UInt32 size)
{
	UInt8		* buf;
	IFileStream	inStream;
	
	buf = (UInt8 *)NewPtr(size);
	if(!buf)
		throw MTMemoryException("Out of memory");
	
	inStream.OpenFile(&rawFile);
	
	inStream.SetPosition(offset);
	inStream.ReadBuffer(buf, size);
	
	inStream.CloseFile();
	
	return buf;
}

//	IFileStream * MTFile::OpenRawFile(void)
//	
//	opens the raw file for streaming
//	
//	returns the stream
IFileStream * MTFile::OpenRawFile(void)
{
	IFileStream	* stream = new IFileStream;
	
	stream->OpenFile(&rawFile);
	
	return stream;
}

//	UInt32 MTFile::LookupIdxFromID(UInt32 id)
//	
//	looks up the index of a file referenced by id
//	
//	id = embedded id
//	returns index, -1 on error
UInt32 MTFile::LookupIdxFromID(UInt32 id)
{
	for(UInt32 i = 0; i < fileList.size(); i++)
		if(fileList[i].id == id)
			return i;
	
	return 0xFFFFFFFF;
}

//	void MTFile::Clear(void)
//	
//	clears the file list
void MTFile::Clear(void)
{
	fileList.clear();
}

//	void MTFile::CopyToFile(IDataStream * in, IDataStream * out, UInt32 size)
//	
//	stream to stream copy
//	
//	in = in stream
//	out = out stream
//	size = bytes to copy
void MTFile::CopyToFile(IDataStream * in, IDataStream * out, UInt32 size)
{
	UInt32	bufSize;
	UInt8	* buf;
	SInt32	left = size;
	
	bufSize = size;
	
	buf = NewSizedPtr(&bufSize);
	
	while(left > 0)
	{
		UInt32	copySize = left;
		
		if(copySize > bufSize)
			copySize = bufSize;
		
		in->ReadBuffer(buf, copySize);
		out->WriteBuffer(buf, copySize);
		
		left -= copySize;
	}
	
	DisposePtr((Ptr)buf);
}

void MTFile::TextDump(void)
{
	std::ofstream	stream;
	char			buf[4096];
	
	stream.open("dump.txt");
	
	stream << "TYPE START    SIZE     ID       NAME\n";
	//         TXMP 00000000 00000000 00000000 file.txmp
	
	for(UInt32 i = 0; i < fileList.size(); i++)
	{
		if(fileList[i].name.length())
			std::sprintf(buf, "%.4s %.8X %.8X %.8X %s", &fileList[i].type, fileList[i].start, fileList[i].size, fileList[i].id, fileList[i].name.c_str());
		else
			std::sprintf(buf, "%.4s %.8X %.8X %.8X", &fileList[i].type, fileList[i].start, fileList[i].size, fileList[i].id);
		
		stream << buf << "\n";
	}
}

//	void MTFile::ParseHeader(IDataStream * stream)
//	
//	parses the database header from a stream
//	
//	stream = database stream
void MTFile::ParseHeader(IDataStream * stream)
{
	stream->SetPosition(0x14);
	fileListElements =	stream->ReadIntelLong();
	nameListElements =	stream->ReadIntelLong();
						stream->Skip(4);
	fileDataOffset =	stream->ReadIntelLong() - 8;
	
	stream->SetPosition(0x28);
	nameTableOffset =	stream->ReadIntelLong();
	
	fileList.reserve(fileListElements);
}

//	void MTFile::ParseFileList(IDataStream * stream)
//	
//	parses the file list from a stream
//	
//	stream = database stream
void MTFile::ParseFileList(IDataStream * stream)
{
	char	buf[4096];
	
	stream->SetPosition(0x40);
	
	for(UInt32 i = 0; i < fileListElements; i++)	// load file table
	{
		UInt32		code, start, nameOffset, size, flags;
		FileEntry	theEntry;
		
		//	flags
		//	
		//	0x00000001 - has no name entry
		
		code =			stream->ReadIntelLong();
		start =			stream->ReadIntelLong();
		nameOffset =	stream->ReadIntelLong();
		size =			stream->ReadIntelLong();
		flags =			stream->ReadIntelLong();
		
		start += fileDataOffset;
		
		theEntry.type =			code;
		theEntry.start =		start;
		theEntry.size =			size;
		theEntry.nameOffset =	nameOffset;
		theEntry.flags =		flags;
		theEntry.name.clear();
		
		fileList.push_back(theEntry);
	}
	
	{	// load file names
		UInt32	tableSize;
		char	* tablePtr = nil;
		
		stream->SetPosition(nameTableOffset);
		tableSize = stream->GetRemaining();
		
		tablePtr = NewPtr(tableSize);
		if(tablePtr)
		{
			stream->ReadBuffer((UInt8 *)tablePtr, tableSize);
			
			for(UInt32 i = 0; i < fileListElements; i++)
			{
				if(!(fileList[i].flags & 0x00000001))
				{
					UInt32	offset = fileList[i].nameOffset + 4;
					UInt8	data = 1;
					char	* bufTraverse = buf;
					
					while((offset < tableSize) && data)
					{
						data = tablePtr[offset];
						
						*bufTraverse++ = data;
						
						offset++;
					}
					
					if(data)
						*bufTraverse = 0;
					
					fileList[i].name = buf;
				}
			}
			
			DisposePtr(tablePtr);
		}
		else	// do slower (non-buffered) method if we can't make a buffer
		{		// then again, if we can't make a buffer we're probably screwed anyway...
			for(UInt32 i = 0; i < fileListElements; i++)
			{
				if(!(fileList[i].flags & 0x00000001))
				{
					stream->SetPosition(nameTableOffset + fileList[i].nameOffset);
					stream->Skip(4);
					stream->ReadString(buf, 4096);
					
					fileList[i].name = buf;
				}
			}
			
			DoError("Memory is very low!");
		}
	}
	
	for(UInt32 i = 0; i < fileListElements; i++)	// load file ids
	{
		if(fileList[i].size >= 4)
		{
			stream->SetPosition(fileList[i].start);
			
			fileList[i].id = stream->ReadIntelLong();
		}
	}
}

//	void MTFile::BuildOtherFiles(void)
//	
//	builds references to other data files
void MTFile::BuildOtherFiles(void)
{
	dataFile = baseFile;
	rawFile = baseFile;
	
	if(dataFile.name[0] < 3)
		throw MTException("Unable to create references for other data files (name too short)");
	
	dataFile.name[dataFile.name[0] - 2] = 's';
	dataFile.name[dataFile.name[0] - 1] = 'e';
	dataFile.name[dataFile.name[0] - 0] = 'p';
	
	baseFile.name[baseFile.name[0] - 2] = 'd';
	baseFile.name[baseFile.name[0] - 1] = 'a';
	baseFile.name[baseFile.name[0] - 0] = 't';
	
	rawFile.name[rawFile.name[0] - 2] = 'r';
	rawFile.name[rawFile.name[0] - 1] = 'a';
	rawFile.name[rawFile.name[0] - 0] = 'w';
}
